classdef myInterfacciaDAQ
    methods (Static)
        %% Interface Graphics (Pre Symulation)- Crea la figure dell'interfaccia e gli assi al suo interno.
        function [F,ax,PL,info]=creaInterfaccia()
            % cosa fa: crea l'interfaccia. Alcuni degli axes e dei plot sono interattivi
            % è una delle prime function ad essere richiamata, inizializza
            % le variabili su cui si basano altre function della libreria.
            % Syntax: creaInterfaccia()
            
            %creazione della figure
            F = figure('OuterPosition',[20 280 750 580],'Color','w',...
                    'Tag','interfaceDAQ'); %,'MenuBar','none'
            
            %asse in alto a sinistra - retro del furgone
            ax{4} = axes(F,'OuterPosition',[0 0.45 0.55 0.55],'Color','w');
            hold(ax{4},'on')
            colorbar(ax{4},'Location','westoutside');
            
            %assi in alto a destra - getti
            ylab = ["top" "bot" "side"];
            for i=1:3 %[1 3 2]
                ax{i} = axes(F,'position',[0.55 1-i*1.5/10 0.4 0.1],...
                    'color','w','YLim',5.5+[0 4],'NextPlot','add',...
                    'YTick',5.5+(0:2:4),'YLim',[5.4 9.6]);
                ax{i}.YLabel.String = ylab(i);
                
                % creazione delle linee anteprima & real-time per l'acquisizione.
                PL.sig(i)=plot(ax{i},nan, nan, '-k','Tag',"lineJet"+i,...
                    'Clipping','on'); %<- anteprima (set it to off)
                hold(ax{i},'on');
                PL.LS(i) = plot(ax{i},nan(1,2),[5.5 9.5],'LineStyle','-','Color','g',...
                                     'LineWidth',2,'Tag',"liveJet"+(i)); %<-real-time
            end
            
            %asse in basso a sinistra - cella di carico.
            ax{5} = axes(F,'OuterPosition',[0 0 0.5 0.4],'Color','w');
            
            PL.LC = animatedline(ax{5},'LineStyle','-','LineWidth',0.5,'Tag','lineLC');
            hold(ax{5},'on')
            PL.ML = animatedline(ax{5},'LineStyle','-','Color',[0.0745    0.6235    1.0000],'LineWidth',0.5,'Tag','meanLC');
            PL.CO = animatedline(ax{5},'LineStyle','-','Color',[1.0000    0.4118    0.1608],'LineWidth',0.5,'Tag','lineLC');
            
            PL.recap = gobjects(2,1);
            PL.recap(1) = plot(ax{5},[nan nan],[nan nan],'--r');
            PL.recap(2) = plot(ax{5},[nan nan],[nan nan],'--b');
            
            ax{5}.YLim = [2.5 4];
            ax{5}.YLimMode = 'auto';
            ax{5}.YLabel.String='Load Cell [V]';
            ax{5}.XLabel.String='samples';
            
            %asse in basso a destra - informazioni utili di simulazione
            ax{6} = axes(F,'OuterPos',[0.47 0.01 0.55 0.45],'Color',[209 247 255]/255);
            testo = @(pos1, pos2, txt) text(ax{6}, pos1,pos2,txt,'Fontweight','bold');
            
            % Colonna a (sinistra)
            info.cd = testo(0.5,4.5, "C_{D.mean}= ?");    hold(ax{6},'on');
            info.cs = testo(0.5,3.5, "LC_{[m., std]}= ?");
            info.ui = testo(0.5,2.5, "U_{\infty} [m/s] = ?");
            info.di = testo(0.5,1.5, "d/L_{truck}  = ?");

            % Colonna a (destra)
            info.c0 = testo(8.5,4.5,"C_{D0}  = ?");
            info.ne = testo(8.5,3.5,"n_{Elica} = ?");
            info.Ap = testo(8.5,2.5,"P_{amb} [Pa] = 97700");
            info.At = testo(8.5,1.5,"T_{amb} [°C] = 19.5");
            
            %lettura dell'icona del furgoncino.
            [img] = imread('truckIcon.png');
            img=double(img);
            img(~img)=nan;  %<- messa in trasparenza (i valori nulli diventano Nan).

            ax{6}.XLim = [0 16];     ax{6}.XColor = 'none'; 
            ax{6}.YLim = [-2 5.5];  ax{6}.YColor = 'none';  
            
            %parametri per il display del furgoncino.(dimensioni e griglia)
            rap = size(img,1)/size(img,2);   L=3.6;
            xg = linspace(0,    L,size(img,2));
            yg = linspace(L*rap,0,size(img,1));
            
            % disegno i furgoncini, 2 fantocci (blu) e 1 arancio (con spie)
            PL.truck{1} = surf(ax{6},xg+8-L/2,yg-1.5,img*0,cat(3,img*0,img*0.5,img*1),'EdgeColor','none');
            PL.truck{2} = surf(ax{6},xg+8-L/2,yg-1.5,img*0,cat(3,img*0,img*0.5,img*1),'EdgeColor','none');
            
            % icona furgone rosso (quello con le spie).
            img(1:10,140:180)=1;
            PL.truck{3} = surf(ax{6},xg+8-L/2,yg-1.5,img*0,cat(3,img,img*0.5,img*0.1),'EdgeColor','none');
            
            % linea di collegamento tra i furgoni.
            PL.truck{4} = plot(L*rap*[1 1]+1,-0.5*[1 1],'-k','Marker','o',...
                'MarkerIndices',[],'MarkerFaceColor','b');
            
            ax{6}.Toolbar.Visible = 'off';
            
            for fn = string(fieldnames(info))'
                switch fn
                  case 'di'
                  info.(fn).ButtonDownFcn = @(src,evt) moveTruck(src,evt);
                  info.(fn).UserData = @(src, PL) ChangeIcons(src, PL);
                  otherwise
                  info.(fn).ButtonDownFcn = @(src,evt) makeEdit(src,evt); 
                end
            end
            
            function makeEdit(src,~)
                src.Editing = 'on';
            end
            
            function moveTruck(src,~)
                src.Editing = 'on';
                waitfor(src,'Editing','off');
                PL=ChangeIcons(src,PL);
            end
            
            function PL=ChangeIcons(src, PL)
                % di default voglio che l'interfaccia mostri sempre le
                % distanze tra i furgoni in modalità percentile ('%')
                [v, st]=myInterfacciaDAQ.interpretStr(src.String, '%');
                src.String = st;
                
                L=3; 
                if size(v,2)>1
                    % due distanze inserite (positive o negative)
                    dis = v(1,:)/100;
                    protag= nnz(dis>0)+1;
                    altro = [1 3 5]; 
                    protag=altro(protag);
                    altro = setdiff(altro, protag);
                    
                    dis = sign(dis).*min(abs(dis),3/length(dis));
                    xx = cumsum([0 1 abs(dis(1)) 1 abs(dis(2)) 1]*L);
                    xoff = (16-xx(end))/2;
                    
                    PL.truck{4}.XData = xoff+[xx(2) xx(3) nan xx(4) xx(5)];
                    PL.truck{4}.YData = -[1 1 nan 1 1];
                    
                    PL.truck{3}.XData = xoff+xg+xx(protag);
                    PL.truck{2}.XData = xoff+xg+xx(altro(1)); %fantoccio
                    PL.truck{1}.XData = xoff+xg+xx(altro(2)); %fantoccio
                    
                    PL.truck{4}.MarkerIndices = [1 2 4 5];
                    
                elseif (size(v,2)==1) && ~isnan(v(1)) 
                    % una sola distanza inserita
                    dis = v(1,1)/100;
                    dis = sign(dis).*min(abs(dis),3/length(dis));
                    PL.truck{4}.XData = 8+[1 -1]/2*dis*L;
                    PL.truck{4}.YData = [-1 -1];
                    PL.truck{4}.MarkerIndices = [1 2];
                    
                    xoff = [-L*(dis<0) -L*(dis>0)];
                    PL.truck{1}.XData = xg+PL.truck{4}.XData(2)+xoff(2);
                    PL.truck{2}.XData = xg+PL.truck{4}.XData(1)+xoff(1);
                    PL.truck{3}.XData = xg+PL.truck{4}.XData(1)+xoff(1);
                else
                    %nessuna distanza leggibile inserita
                    xoff = (16-L)/2;
                    PL.truck{3}.XData = xoff+xg;
                    PL.truck{2}.XData = xoff+xg;
                    PL.truck{1}.XData = xoff+xg;
                    
                    PL.truck{4}.XData = nan;
                    PL.truck{4}.YData = nan;
                    PL.truck{4}.MarkerIndices = [];
                end
            end
        end

        %% Interface Graphics (Pre Symulation)- Popola l'asse #4 con scatter e surf (<-il retro del furgone)
        function [Mic,updMic,setCaxis]=preparaRear(asse)
            % cosa fa: crea una visualizzazione dei microfoni sul retro
            % furgone (può essere usata anche al di fuori
            % dell'interfaccia).
            % Crea e fornisce in output lo scatter dei 16
            % microfoni (Mic), un function handle per aggiornare il plot
            % quando i loro valori cambiano (updMic), e un function handle
            % per modificare la scala cromatica sempre cambiando i loro
            % valori (setCaxis).
            % Syntaxes:
            %   preparaRear( ax{4} )
            %       updMic( vettore [1 x 16] )
            %       setCaxis( vettore [1 x 16] )
            
            %coordinate dei microfoni.
            %gentilmente passati da Enrico:
            x_probe = [nan(1,4), 106.16 85.5 135.19 95.19 135.19 95.19 135.19 95.19 135.19 95.19 106.16 85.5]; %Microphones' positions in x
            y_probe = [nan(1,4), 165.95 165.95 135.5 135.5 110.5 110.5 65.5 65.5 90.5 90.5 35.7 35.7];         %Microphones' positions in y
            L = length(x_probe);
            
            h_wheel = 10;
            x_probe = x_probe-x_probe(6);  % il microfono (6) è quello al centro.
            
            larg = 171;
            alte = 210 -h_wheel;
            
            m = mean(x_probe(5:6));
            x_probe(1:4) = [-larg/2 m larg/2 m];
            
            m = mean(y_probe([10 14]));
            y_probe(1:4) = [m alte m 0];
            y_probe = y_probe+h_wheel;
            
            rectangle(asse,'position',[-larg/2 h_wheel larg alte],'Curvature',[0.05 0.05])
            hold(asse,'on')
            c = 0.3;
            rectangle(asse,'position',[-larg/2   0 20 h_wheel],'Curvature',[c c])
            rectangle(asse,'position',[larg/2-20 0 20 h_wheel],'Curvature',[c c])
            
            plot(asse,[0 0 nan -larg/2*1.1 larg/2*1.1],[0 alte nan alte*[.5 .5]]+h_wheel,'--r','LineW',0.25)
            Mic = scatter(asse,x_probe, y_probe, 15, zeros(1,L), 'o',...
                      'filled','MarkerEdgeColor','k');
                  
            Mic.UserData.broken = [1 2 3]; % <--indici dei microfoni rotti
             
            updMic.col = @(v) set(Mic,'CData',v);
            updMic.siz = @(v) set(Mic,'SizeData',ceil(abs(v)));
            asse.DataAspectRatio = [1 1 1/100];
            
            setCaxis = @(v) caxis(asse, v);
            
            asse.XColor = 'none';
            asse.YColor = 'none';
            %asse.Title.String = 'Cd=? U_{\infty}=?';          
        end
        
        %% Interface Graphics (Pre Symulation)- Surface del retro furgone interpolata dallo Scatter Microphones
        function [S,updSurf]=addSurf(x,y,res)
            % cosa fa: crea la superficie (S) che si estende su tutto il
            % retro del furgone e cerca di interpolare i 16 valori dei
            % microfoni. fornisce in output anche un function handle
            % (updSurf) per aggiornare la superficie S al variare dei
            % valori dei microfoni. La superficie è 3D.
            % in input è necessario specificare la posizione dei microfoni
            % (x,y) mentre la risoluzione della griglia dipende da (res).
            % addSurf richiama -> howToInterp, che richiama -> gridfit.m,
            % quest'ultima è una funzione di interpolazione più performante
            % di interp2.
            % Syntaxes:
            % -  [S, updSurf]=addSurf( x_micVet, y_micVet, (intero > 10) )
            % -      updSurf( micValues [1 x 16])
            
            global ax
            
            %creo la griglia.
            xv=linspace(x(1),x(3),res);
            yv=linspace(y(4),y(2),res);
            [xgrid,ygrid]=meshgrid(xv,yv);
            
            %creo la surf, con una trasparenza del 50% (AlphaData)
            S=surf(ax{4}, xgrid, ygrid,zeros(res,res), nan(res,res),...
                'EdgeC','none','Tag','MicSurf','AlphaData',0.5);
            
            %modifico la gerarchia dei plot sull'asse per mettere la
            %superficie in background.
            ax{4}.Children = ax{4}.Children([2:end 1]);
            
            %creo il function handle che possa interpolare i valori su
            %tutta la griglia (xv,yv) a partire da un set di nuovi valori 'z'
            %nei soli punti (x,y).
            fcn = myInterfacciaDAQ.howToInterp(x,y,xv,yv);
            updSurf = @(zVet) fcn(zVet);  
        end
        
        %% Interface Graphics (Pre Symulation)- Interpolation method for Microphone surface
        function fcn = howToInterp(x,y,xv,yv) 
            % cosa fa: howToInterp è richiamata da "addSurf(..)", e
            % restituisce un function handle. Il function handle dovrebbe
            % essere comandato fornendo un vettore di valori qualsiasi, 
            % per poi interpolare quei valori, posizionati su un set non
            % regolare di punti (x,y) vettori, su tutta una griglia (xv,yv)
            % vettori.
            % ipotesi (1)
            % Inoltre considerando che alcune spie nel caso in questione
            % sono funzionanti ma la loro misurazione non è attendibile, in
            % prima battuta howToInterp esclude tali spie dal feed.
            % ipotesi (2)
            % dal momento che i microfoni sono stati posizionati solo sulla
            % metà destra del retro furgone, il campo è stato (per
            % semplicità) reso speculare anche sulla parte sinistra.
            
            broken = 1:3;  % <-- microfoni rotti, da escludere.
            
            % ottengo gli indici della porzione simmetrica, i microfoni 6 e
            % 16 sono esattamente sulla linea di simmetria: non vanno
            % inclusi per evitare ripetizioni.
            mirror = setdiff(1:16,[6 16 broken]);  
            
            xmir = -x(mirror); % <-- imposto una controparte simmetrica
            ymir =  y(mirror);
            x(broken) = []; %<-- elimino i punti con microfoni rotti.
            y(broken) = [];
            
            left = setdiff(1:16, broken); %<-- gli indici a meno dei mic rotti.
            x = [xmir x];  %<-- unisco tutti i punti buoni.
            y = [ymir y];  
            
            % la function di interpolazione per un qualunque input (zVet)
            fcn = @(zVet) gridfit(x,y,zVet([mirror left]), xv, yv);
        end
        
        %% Interface Graphics (Pre Symulation)- Plot dei segnali getto by matrix
        function plotJet( axList, mat, f, valid)
            % cosa fa: rappresenta sull'interfaccia il segnale dei getti.
            % è necessario specificare:
            % 1) (axList)   la lista dei 3 assi.
            % 2) (mat)      la matrice [N x 3] rappresentante i segnali.
            % 3) (f)        la frequenza di trasmissione con il sistema DAQ
            % 4) (valid)    facoltativo, un booleano che modifica il colore
            %           del plot, per distinguere tra le modalità (on / off).
            %Syntaxes:
            %   plotJet( ax(1:3), mat [N x 3], f (scalar))
            %   plotJet( ax(1:3), mat [N x 3], f (scalar), true/false)
            
            global PL
            % se valid non è specificato, si da per scontato che i getti
            % siano in modalità off.
            if nargin<4
               valid = false; 
            end
            
            % leggo il numero di samples dei segnali.
            L = size(mat,1);
            
            % la variabile adibita al plot dei 3 getti sarebbe "PL.sig",
            % quindi se la property 'sig' non è presente in PL, allora devo
            % richiamare la funzione plot.
            if ~isfield(PL,'sig')
                
                % ogni segnale viene plottato nel diagramma (time, Voltage)
                for i=1:size(mat,2)
                PL.sig(i)=plot(axList{i},(0:L-1)'/f, mat(:,i), '-k','Tag',"lineJet"+i);
                 if ~valid
                  % se attivo, è nero, se non attivo, grigio.
                  PL.sig(i).Color = [1 1 1]*0.5;
                 end
                end
            else
            % nel caso in cui i plot siano già stati creati, non è necessario
            % utilizzare 'plot', ci si limita ad aggiornarne i punti.
                for i=1:size(mat,2)
                PL.sig(i).XData = (0:L-1)'/f;
                PL.sig(i).YData = mat(:,i);
                end
            end
        end
        
        %% Interface Simulation (On Symulation)- Write data to *.txt file and plot data (if requested)
        function writeOnFile(source, ~, fileID, plotFcn)
            % cosa fa: fornita in input una sorgente (source, a scelta tra
            % dm, dg, dq) e un puntatore a file *.txt (fileID), writeOnFile
            % legge i dati nella cache della sorgente, per tutti i canali
            % che lo strumento possiede, e poi li scrive nel file,
            % specificando anche l'istante temporale di acquisizione.
            % Oltre alla lettura e scrittura, si esegue poi un function
            % handle fornito in input (plotFcn) che prende in carico i
            % valori della porzione letta, in modo da diversificare il
            % rendering a seconda del canale (getto/cella di carico/microfono).
            % (!) Attenzione: questa function viene eseguita più e più
            % volte, non esegue la lettura integrale di tutta la
            % simulazione in un colpo solo.
            
            %Lettura parziale dallo strumento.
            [data, timestamps,~] = read(source, source.ScansAvailableFcnCount, "OutputFormat", "Matrix");
            
            %Scrittura su file.
            formato = ['%f' repmat('\t%f',1,size(data,2)) '\n'];
            fprintf( fileID, formato, [timestamps data]');
            
            % display sulla command window
            % disp([timestamps(end) data(end,:)]);
              
            %Esecuzione di un aggiornamento grafico, con i valori contingenti.
            plotFcn(timestamps, data);
            
        end
        
        %% Interface Graphics (On Symulation)- Update "Load Cell" plot by signal acquired
        function updLC( t, dQsignal) 
            %(?) cosa fa: updLC(t, dQsignal) aggiorna il plot dei segnali della
            %   cella di carico e del contraction factor del tubo di pitot. è
            %   necessatio fornire il vettore degli istanti temporali (t) e
            %   una matrice di altrettante righe con 4 colonne, di cui solo la
            %   1° e la 4° sono di particolare interesse.
            % (*) Nota: updLC inoltre aggiorna il titolo, riportando
            %   il valor medio del segnale(voltage) della cella di carico
            %   contingente. Inoltre, se esistono i plot dei segnali getto
            %   (PL.LS) aggiorna la posizione di una barra che facilita la visualizzazione
            %   del segnale che proprio in quel momento dovrebbe essere eseguito.
            % (E) edit: man mano che la simulazione procede, tengo il conto
            %    del valor medio del segnale che registro.
            global PL ax
            
            if isvalid(PL.LC)
                %Rendering dei segnali CELLA di CARICO e CONTRACTION FACTOR.
                %   addpoints è una funzione built-in di matlab che aggiorna
                %   le animated lines "PL.LC" ed "PL.CO". 
                PL.LC.addpoints(t,dQsignal(:,1));
                PL.CO.addpoints(t,dQsignal(:,4))
                %ax{5}.YLim = 'auto';
                
                %Rendering del segnale mediato della cella di carico.
                m = mean(dQsignal(:,1));
                PL.ML.addpoints(mean(t([1 end])),m);

                %Rendering del titolo (valor medio della cella)
                ax{5}.Title.String = "Actual Mean Value: "+string(m);
            end
            
            %Rendering delle barre sulla vista dei segnali Getto.
            for i=1:3
                if isvalid(PL.LS(i))
                 PL.LS(i).XData = [1 1]*t(end);
                end
            end
        end
        
        %% Interface Graphics (On Symulation)- Update Microphones by signal acquired
        function updMI( dMsignal, zetaChange, surfChange, varargin)
            %cosa fa: se la function di libreria 'howToInterp' si preoccupa
            %di calcolare tutti i valori Z di pressione della griglia del
            %retro furgone, è la function updMI che si preoccupara di
            %renderizzare il risultato sullo schermo.
            % input list:
            %   1) (dMsignal) una matrice dei 16 valori microfonici
            %        in un numero T di istanti [T x N].
            %   2) (zetaChange) booleano, se true, la superficie diventa 3D
            %        e la visuale si adatta di conseguenza. La profondità è
            %        associata ai valori dMsignal o a un loro derivato.
            %   3) (surfChange) un function handle che renderizzi la
            %        superficie a partire dai valori (dMsignal) o un loro
            %        derivato, se questo input è vuoto [] la superficie
            %        rimane invisibile, e il pc fatica di meno.
            %   4) + un input facoltativo (process), deve prendere in input
            %        una matrice come dMsignal e far uscire fuori un
            %        vettore con 16 canali. Utile per eseguire la media o
            %        l'rms o la std del pacchetto misurato.
            %         Se si omette questo input, vale la regola per cui è
            %         l'ultimo scan ad essere renderizzato sullo schermo.
            %        
            %Syntaxes:
            % 1) per nascondere la superficie (mantenendo lo scatter)
            %  updMI( s, 0/1, [])
            % 2) per mostrare la superficie (2D)
            %  updMI( s, 0, updSurf)
            % 3) per mostrare la superficie (3D)
            %  updMI( s, 1, updSurf)
            % 4) per applicare rendering con il valor medio del segnale
            %  updMI( s, 0/1, []/upSurf, @mean )
            
            global PL ax
            
            broken = PL.Mic.UserData.broken;
            filtroBroken = true(1,16);
            filtroBroken(broken) = false;
            
            % Solamente se lo scatter dei microfoni esiste eseguo le operazioni!
            if isvalid(PL.Mic)
            % 0) se non dico come elaborare il pacchetto, prendo solo valori
            % di pressione (16 microfoni) relativi all'ultimo istante del pacchetto appena scansionato. 
                process = @(x) x(end,:);
                if ~isempty(varargin)
                    if contains(class(varargin{1}),'handle')
                        process = varargin{1}; %<- la funzione dell'utente!
                    end
                end
                
            % 1) applico l'operazione al pacchetto. 
            vals = process(dMsignal); 
            sizes = abs(vals)*500;
            sizes(~filtroBroken) = nan;
            
            % 2) modifico i colori (CData) e le dimensioni (SizeData) dei pallini
            %dello scatter.
            PL.Mic.CData = vals;
            PL.Mic.SizeData = sizes;
            
            % 3) aggiorno il fondoscala della colorbar
            [mm, MM] = bounds(vals(filtroBroken),'all');
            caxis(ax{4},[mm MM]);
            
                % 4) per modificare la surface del RearTruck, ecco le
                % istruzioni:
                % se ho un FcnHandle per la surface, lo applico!
                if contains(class(surfChange),'handle')
                    PL.SurfM.CData = surfChange(vals);
                   
                    %4A) Se devo aggiornare anche la dimensione Z per il 3D,
                    %applico gli stessi valori di CData a ZData
                    if zetaChange
                        PL.SurfM.ZData = PL.SurfM.CData;
                    end
                    
                    %Se ho applicato un update surface, mi assicuro che sia
                    %Visible=true
                    if ~PL.SurfM.Visible
                       PL.SurfM.Visible = 'on'; 
                    end
                else
                    %4B) Se non ho un update da applicare, nascondo la surface.
                    PL.SurfM.Visible = false;
                end
                
                %5) Che io voglia applicare o meno updates sulla surface, devo
                %modificare anche gli ZData dello scatter:
                if zetaChange
                    %5A) Applico gli ZData e sposto la visuale in 3D nel caso
                    %sia ancora in 2D (<--view=[0 90])
                    PL.Mic.ZData = vals;
                    if (ax{4}.View(2)==90)
                       view(ax{4},3);
                    end
                else
                    
                   %5B) Se invece non voglio avere una animazione 3D, disattivo
                   %tutto: riportando i dati a 0 e con la visuale in 2D.
                   if isempty(PL.Mic.ZData) || PL.Mic.ZData(1)
                   PL.Mic.ZData   = zeros(size(PL.Mic.XData));
                   PL.SurfM.ZData = zeros(size(PL.SurfM.XData));
                   end
                   
                   if ax{4}.View(2)~=90
                      view(ax{4},2);
                   end
                end
                
            end
        end

        %% Interface Graphics (Pre Symulation)- pulizia dei plot
        function resetPlots(durata,valid)
            %   cosa fa: resetPlots(durata,valid) è una function che viene
            %lanciata a cavallo tra due simulazioni, e serve ad eliminare
            %ciò che i plot mostrano della vecchia simulazione per fare
            %posto ai nuovi dati.
            % (durata) deve essere il valore (in secondi) della nuova
            % simulazione che si intende effettuare.
            % (valid) deve essere la tripletta di booleani che determina se
            % si desidera utilizzare i rispettivi strumenti:
            % cella/microfoni/getti.
            %   cosa pulisce: i segnali della cella di carico, del contraction
            %factor, + gli eventuali valori medi della tensione tra caso
            %forzato e non forzato.
            % Syntaxes: 
            % - resetPlots( 90, [0 1 1])
            % - resetPlots( 60, [1 1 0])
               
            global ax PL
            clearpoints(PL.LC); % <-pulizia segnale cella di carico
            clearpoints(PL.ML); % <-        cella di carico a valor medio
            clearpoints(PL.CO); % <-        contraction factor
            for i=1:2
            PL.recap(i).XData = [nan nan]; % <- valori medi tratteggiati di 
            PL.recap(i).YData = [nan nan];
            end
            
            if nargin==2
               if valid(1)
                  PL.LC.Color = 'k'; PL.ML.Color='g';
               else
                  PL.LC.Color = [1 1 1]*0.5; PL.ML.Color='g'; 
               end
            end
            
            ax{4}.Title.String = 'Waiting to start';
            for i=[1:3 5] %reset degli assi Getto e Cella di carico
                ax{i}.XLim = [0 durata]+[-.1 .1];
                if i<4
                    if isvalid(PL.LS(i))
                   PL.LS(i).XData = nan(1,2); %pulizia segnale Getti
                   if nargin==2
                      if valid(3)
                         PL.LS(i).Color = 'g';
                      else
                         PL.LS(i).Color = [1 1 1]*0.3;
                      end
                   end
                    end
                end
            end 
        end
        
        %% Interface Graphics (Post Symulation) - Save Notes presenti su interfaccia in asse #6
        function saveNotes(baseFileName)
        %cosa fa: saveNotes(baseFileName) legge dall'interfaccia i log
        %  displayati nell'asse #6, contenente informazioni su CD, CD0, Uinf
        %  eccetera.. e scrive tutte quante le informazioni sul file dedicato.
        %  Non è mecessario che l'utente compili tutti i log, i più
        %  difficili vengono displayati in automatico alla fine della
        %  simulazione da soli. Gli unici che il software non è in grado di
        %  conoscere sono il numero di giri dell'elica e le distanze d/L
        %  dei furgoni. SaveNotes si attiva di solito nelle fasi finali di 'GOsym(..)',
        %  quindi l'utente ha tutto il tempo a disposizione per compilare,
        %  se la simulazione è sufficientemente lunga.
        %(!) attenzione: la stringa (baseName) in input deve specificare il percorso
        %  completo del PC. Inoltre se in tale stringa è presente una 'X',
        %  questo carattere viene sostituito con la tag '_Notes' come da procedura standard.
        %  I log che presentano il carattere '?' (cioè che non vengono
        %  compilati) vengono scritti sul file con l'assegnazione 'nan'.
            
            global ax
            
            if isvalid(ax{6})
                % aggiunta della Tag '_Notes' e del formato '.txt'
                baseFileName = replace(baseFileName,'X','_Notes');
                if ~endsWith(baseFileName,'.txt')
                    baseFileName = baseFileName+".txt";
                end
                
                % apertura, scrittura e chiusura del file di logs.
                f = fopen(baseFileName,'w');
                for i=(ax{6}.Children)' 
                    if contains(class(i),'text','IgnoreCase',true)
                        contenuto = replace(i.String,'?','nan');
                        fprintf(f,'%s\n',contenuto);
                    end
                end
                fclose(f);
            else
                disp('Cannot save notes')
            end
        end
        
        %% Interface Graphics (Pre Symulation) - Connetti il PC ai moduli National INstruments
        function [dq,dm,dg] = connect2DAQ(realConn,showInfoModules)
            %cosa fa: connect2DAQ crea e fornisce in input 3 struct con cui interagire
            % con la strumentazione hardware, rispettivamente:
            % (dq) - per la cella di carico e il tubo di pitot.
            % (dm) - per i 16 microfoni.
            % (dg) - per i tre getti (top/side/bot).
            % In input ha bisogno di
            %   (realConn) un booleano che consente di creare realmente la
            % connessione con il sistema DAQ (se è true); oppure di creare
            % delle dq, dm, dg fittizie nel caso non si abbia intenzione di
            % eseguire misurazioni, (se è false).
            %   (showInfoModules) un booleano che mostra le informazioni
            %   tecniche della strumentazione DAQ in utilizzo.
            %Syntaxes:
            % (realmente connesso) [dq,dm,dg]= connect2DAQ( true, t/f )
            % (disconnesso       ) [dq,dm,dg]= connect2DAQ( false , t/f)
           
            if  realConn %<-- realmente connesso al DAQ.
                % I TRE GETTI: cDAQ1Mod8/ao0:2
                
                % TEST READ:   cDAQ1Mod1/ai0:3, cDAQ1Mod2/ai0:3, cDAQ1Mod3/ai0:3, cDAQ1Mod4/ai0:3,
                % cDAQ1Mod6/ai0:1, <-- cella di carico, il canale ai0 (modulo 6)
                % cDAQ1Mod5/ai0:1  <-- contracion factor, il canale ai1 (modulo 5)
                
                % microfoni non funzionanti: 1 2 3
                % -capsule ZOC non funzionanti: 30-
                
                % NON RIESCI  A LEGGERLI? HAI ATTACCATO IL CAVO USB ?
                % APPLICA SULLA COMMAND L'ISTRUZIONE: daqreset
                daqreset
                
                dL = daqlist("ni");
                
                switch showInfoModules
                    case true
                        for i = 1:length(dL)
                            disp("-----("+i+")-----")
                            deviceInfo = dL{ i, "DeviceInfo"};
                            disp(deviceInfo);
                        end
                    case false
                end
                dq = daq('ni'); %istanza di connessione per: Cella di carico, +altri 2 parametri, +Contraction factor
                dg = daq('ni'); %istanza di connessione per: 3 canali output (getti bot/side/top).
                dm = daq('ni'); %istanza di connessione per: 16 microfoni
                
                %(A) Inserimento canali per Cella di carico in dQ----------
                % Esempio di comando per aggiungere il canale:
                %   addinput(dq, "cDAQ2Mod6", "ai0", "Voltage");
                module = "cDAQ2Mod";
                tipo   = "Voltage";
                
                for canale = module+[6 5]
                    for ai = "ai"+(0:1)
                        addinput(dq, canale, ai, tipo);
                    end
                end
                dq.Channels
                
                %(B) Inserimento 16 canali per Microfoni in dM-------------
                for canale = module+(1:4)
                    for ai = "ai"+(0:3)
                        addinput(dm, canale, ai, tipo);
                    end
                end
                dm.Channels
                
                %(C) Inserimento 3 canali per Getti in dG------------------
                module = "cDAQ2Mod8";
                tipo = "Voltage";
                addoutput(dg, module,"ao0",tipo); % top
                addoutput(dg, module,"ao1",tipo); % bot
                addoutput(dg, module,"ao2",tipo); % side (leggere il pannello in legno delle elettroValvole Lab)
                % /non serve il 4°/-> addoutput(dg,gettoInfo1,"ao3","Voltage");
                
                dg.Channels
            else %realConn = false <-- non sei connesso, ma vuoi vedere cosa succede.
                dg=struct;  dg.Channels=[1 1];
                dm=dg; dq=dg;
                
            % Le function write, preload, start e flush semplicemente
            % eseguono dei disp('message') solo per verificare che le
            % variabili dm, dq, dg si comportino correttamente nel processo principale. 
            
                dg.preload = @(x) preload(x);
                dg.write = @(x) write(x);
                
                dm.flush = @(x) flush(x);
                dm.start = @(x) start(x);
                dg.flush = @(x) flush(x);
                dg.start = @(x) start(x);
                dq.flush = @(x) flush(x);
                dq.start = @(x) start(x);
                
            end
           
                function write(varargin)
                disp('Fake Device: write')
                end
                
                function preload(varargin)
                disp('Fake Device: preload')
                end
                
                function start(varargin)
                disp('Fake Device: start')
                end
                
                function flush(varargin)
                disp('Fake Device: flush')
                end
        end
        
        %% Interface Graphics (On Symulation) - Al termine della simulazione, aggiorna i dati
        function ReportOnUI(out)
            %cosa fa: al termine di una simulazione, è necessario
            %aggiornare graficamente alcune parti dell'interfaccia, in
            %particolar modo i display e i segnali medi di tensione della cella di carico.
            %(out) deve essere una struct con queste properties:
            %   out.CD -> il CD calcolato nella 2° fase della misurazione.
            %   out.m  -> valor medio di tutto il segnale (voltage) cella
            %               di carico LC
            %   out.st -> standard deviation di LC (voltage)
            %   out.U  -> velocità del vento in galleria (1° fase quello di rif.)
            %   out.CD0 -> il CD calcolato nella 1° fase della misurazione,
            %               senza forcing
            %   out.tempo -> tempo (in secondi) di tutta la simulazione
            %   out.del   -> tempo (in secondi) di fine 1° fase ed inizio
            %               forcing
            %   out.noF e out.siF -> struct per 1° e 2° fase, senza e con
            %               forcing, entrambe contenenti (.LC)
            
            global info ax PL
            rr = @(x) round(x,2); %<- funzione di arrotondamento alla 2° cifra decimale.
            info.cd.String = "C_{D.mean}= "+out.CD;
            info.cs.String = "LC_{[m., std]}= ["+ rr(out.m) +"; "+rr(out.st)+ "]";
            info.ui.String =  "U_{\infty} [m/s] = "+rr(out.U);
            info.c0.String = "C_{D0}  = "+out.CD0;
            
            ax{1}.Parent.Name = "Simulazione Conclusa";
            
            PL.recap(1).XData = [0          out.del];
            PL.recap(2).XData = [out.del    out.tempo];
            
            PL.recap(1).YData = [1 1]*out.noF.LC;
            PL.recap(2).YData = [1 1]*out.siF.LC;
            
            dcd = (out.CD-out.CD0)/out.CD0;
            ax{5}.Title.String = sprintf('\\DeltaCD = %+.1f%% (%+.4f)', dcd*100,out.CD-out.CD0);
        end
        
        %% Interface Graphics (On Symulation) - Fa Partire la Simulazione! :)
        function out = GOsym(channelValid, aux)
        % cosa fa: GOsym organizza l'esecuzione elementare di una
        %   simulazione (leggere i commenti interni per le varie fasi).
        % cosa produce: produce (out), una struct contenente svariati
        %  calcoli racchiusi nelle rispettive properties, tra cui i nomi
        %  completi dei file, dopo che sono stati scritti e spostati nelle
        %  directory dedicate (out.baseName).
        % input list: (channelValid) una tripla di booleani per
        %   indicare quali tra i rispettivi canali LoadCell, Mic, Getto,
        %   sono effettivamente attivi. 
        %   (aux) è una struct composta quanto segue:
        %       (aux).tempo  [s] durata di simulazione
        %       (aux).calculation.delay  [s] durata fase senza forcing
        %       (aux).calculation.Zero [1 x 2] valori di taratura.
        %       (aux).baseName struct contenente i nomi dei file Loa Get Mic e Note (consulta myOutputFile)
        %            *di solito questi file sono creati sulla directory principale, e poi migrati in una cartella subordinata
        %       (aux).fileDest struct contenente i puntatori ai file associati al *.baseName
        %       (aux).newPath  [string] è la directory verso cui i file saranno poi spostati. 
        % Syntax: 
        %   out = GOsym( [1 1 1], aux [struct])   
            
            global ax dg dq dm
            
            % Richiamo le librerie di supporto.
            LibI = myInterfacciaDAQ;
            LibO = myOutputFile;
            LibT = myTdmsReader;
            
            % Se almeno un canale è connesso ed è un canale hardware.
            if any(channelValid) && any(cellfun(@LibO.isDevice,{dq,dm,dg}))
                if isvalid(ax{4})
                    ax{4}.Title.String = 'Running';
                end
                % tic
                
                % Imposto la durata di simulazione.
                durata = seconds(aux.tempo);
                lookAheadWindow = aux.tempo;    
                
                % Prima di cominciare, aspetto 2 secondi cautelativi.
                uiwait(ax{1}.Parent, 2);
                
                % Per ogni canale connesso e abilitato, faccio partire l'esecuzione.
                % (Intanto Matlab continua ad eseguire il codice sotto, non attende)
                for i=1:3
                    if channelValid(i)
                        switch i
                            case 1
                                start(dq,"Duration",durata)
                            case 2
                                start(dm,"Duration",durata)
                            case 3
                                dg.start;
                        end
                    end
                end
                % Aspetto il termine della simulazione.
                uiwait(ax{1}.Parent, lookAheadWindow)
                
                % Aspetto altri 1 secondi cautelativi.
                uiwait(ax{1}.Parent, 1);
                ax{4}.Title.String = 'Simulation Over';
                
                % Chiusura dei devices e dei file
                LibO.deviceOff(channelValid, aux.fileDest); 
                
                %Faccio due rapidi conti, e aggiorno i logs
                %sull'interfaccia con 'ReportOnUI'
                if channelValid(1) && isfield(aux, 'calculation')
                    out = LibT.DoTheMath(  channelValid, aux.tempo, aux.calculation.delay,...
                                                    aux.calculation.Zero);
                    LibI.ReportOnUI(out);
                else
                    out = struct();
                end
                
                % salvo le scritte dei logs.
                LibI.saveNotes(aux.baseName.Note);
                
                %Spostamento dei file in una directory interna a quella principale.
                out.baseName = LibO.migraFile(aux.baseName, aux.newPath);
                fprintf('\tAcquisizione conclusa!\n')
            else
                disp('Acquisizione non permessa: nessun canale abilitato')
                out = struct();
            end
            
        end
        
        %% Interface Graphics (Pre Symulation) - Legge la distanza tra furgoni inserita da utente
        function [v, premessa]=interpretStr(str, mode)
            % cosa fa: data una stringa (str) contenente una o più distanze dei
            % convogli rispetto al furgone con le spie installate, questa
            % è in grado di estrapolarne i numeri ivi inseriti, e di
            % convertire la stringa in uscita nel formato (mode) desiderato,
            % scegliendo tra('%' oppure '[m]').
            % (in output i valori in percentuale [%] sulla 1° riga, 
            % reale [metri] sulla 2° riga).
            % se la stringa in input non presenta numeri leggibili,
            % restituisce in output valori nan e stringa immutata.
            % Syntaxes
            % [ditanze, stringa ] = interpretStr( '12% -3% ','%' oppure '[m]')
            % [ditanze, stringa ] = interpretStr( '1.3 -2.0','%' oppure '[m]')
            
            premessa = 'd/L_{truck}  = ';
            L = 0.41; % lunghezza del furgone [m]
            
            %estraggo la distanza
            num = regexp(str,'[-\d\.]*','match');
            
            % verifico che la distanza sia già espressa in [%] o meno.
            hasPerc = contains(str,'%');
            
            v = nan;
            if iscell(num) && ~isempty(num)
                v = str2double(num);
            end
            if ~isempty(v) && ~any(isnan(v)) 
                % converto tutte le distanze da stringhe a numeri.
                % mi aspetto di ottenere una riga di valori.
                v = str2double(num);
                
                %per una corretta rappresentazione, devo dare la priorità
                %alle distanze positive (quelle davanti al furgone) e poi a
                %quelle negative (quelle che seguono il furgone)
                [~,ord]=sort(sign(v),'descend');
                v = v(ord);
                
                %creo una matrice di due righe, la prima per i valori in
                %formato percentuale [%], la seconda con i valori nominali [m].
                if hasPerc
                    v   = v.*[1; L/100];    %[(%); (m)]
                else
                    v   = v.*[100/L; 1];    %[(%); (m)]
                end
                
                v(1,:) = round(v(1,:),0);
                v(2,:) = round(v(2,:),3);
                
                if strcmp(mode,'%')
                    premessa = premessa+strjoin( ""+v(1,:)+"%",'  ');
                else
                    premessa = premessa+strjoin( ""+v(2,:),    '  ');
                end
                premessa = char(premessa);
            else
                v = [nan; nan];
            end
        end
    end
end

%%